<?php
require_once 'config.php';
requireLogin();

$database = new Database();
$db = $database->getConnection();

$message = '';
$message_type = '';

// Verificar si las nuevas columnas/tablas existen
$has_estado = false;
$has_observaciones = false;
$has_etiquetas = false;
$has_historial = false;

try {
    $check = $db->query("SHOW COLUMNS FROM contactos LIKE 'estado'");
    $has_estado = $check->rowCount() > 0;
    
    $check = $db->query("SHOW COLUMNS FROM contactos LIKE 'observaciones'");
    $has_observaciones = $check->rowCount() > 0;
    
    $check = $db->query("SHOW TABLES LIKE 'etiquetas'");
    $has_etiquetas = $check->rowCount() > 0;
    
    $check = $db->query("SHOW TABLES LIKE 'historial_cambios'");
    $has_historial = $check->rowCount() > 0;
} catch (Exception $e) {
    // Ignorar errores de verificación
}

// Función para registrar cambios en el historial (solo si la tabla existe)
function registrarCambio($db, $contacto_id, $usuario_id, $campo, $valor_anterior, $valor_nuevo, $has_historial) {
    if ($has_historial && $valor_anterior != $valor_nuevo) {
        try {
            $query = "INSERT INTO historial_cambios (contacto_id, usuario_id, campo_modificado, valor_anterior, valor_nuevo) VALUES (?, ?, ?, ?, ?)";
            $stmt = $db->prepare($query);
            $stmt->execute([$contacto_id, $usuario_id, $campo, $valor_anterior, $valor_nuevo]);
        } catch (Exception $e) {
            // Ignorar errores de historial
        }
    }
}

// Procesar acciones AJAX
if (isset($_POST['ajax_action'])) {
    header('Content-Type: application/json');
    
    if ($_POST['ajax_action'] == 'get_historial' && $has_historial) {
        try {
            $contacto_id = (int)$_POST['contacto_id'];
            $query = "SELECT h.*, u.nombre as usuario_nombre FROM historial_cambios h 
                      LEFT JOIN usuarios u ON h.usuario_id = u.id 
                      WHERE h.contacto_id = ? ORDER BY h.fecha_cambio DESC";
            $stmt = $db->prepare($query);
            $stmt->execute([$contacto_id]);
            $historial = $stmt->fetchAll();
            echo json_encode($historial);
        } catch (Exception $e) {
            echo json_encode([]);
        }
        exit;
    }
    
    if ($_POST['ajax_action'] == 'update_estado' && $has_estado) {
        try {
            $contacto_id = (int)$_POST['contacto_id'];
            $nuevo_estado = $_POST['estado'];
            
            // Obtener estado anterior
            $query = "SELECT estado FROM contactos WHERE id = ?";
            $stmt = $db->prepare($query);
            $stmt->execute([$contacto_id]);
            $estado_anterior = $stmt->fetchColumn();
            
            // Actualizar estado
            $query = "UPDATE contactos SET estado = ? WHERE id = ?";
            $stmt = $db->prepare($query);
            if ($stmt->execute([$nuevo_estado, $contacto_id])) {
                registrarCambio($db, $contacto_id, $_SESSION['user_id'], 'Estado', $estado_anterior, $nuevo_estado, $has_historial);
                echo json_encode(['success' => true]);
            } else {
                echo json_encode(['success' => false]);
            }
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
        exit;
    }
    
    if ($_POST['ajax_action'] == 'update_observaciones' && $has_observaciones) {
        try {
            $contacto_id = (int)$_POST['contacto_id'];
            $nuevas_observaciones = $_POST['observaciones'];
            
            // Obtener observaciones anteriores
            $query = "SELECT observaciones FROM contactos WHERE id = ?";
            $stmt = $db->prepare($query);
            $stmt->execute([$contacto_id]);
            $observaciones_anteriores = $stmt->fetchColumn();
            
            // Actualizar observaciones
            $query = "UPDATE contactos SET observaciones = ? WHERE id = ?";
            $stmt = $db->prepare($query);
            if ($stmt->execute([$nuevas_observaciones, $contacto_id])) {
                registrarCambio($db, $contacto_id, $_SESSION['user_id'], 'Observaciones', $observaciones_anteriores, $nuevas_observaciones, $has_historial);
                echo json_encode(['success' => true]);
            } else {
                echo json_encode(['success' => false]);
            }
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'error' => $e->getMessage()]);
        }
        exit;
    }
    
    echo json_encode(['success' => false, 'error' => 'Acción no disponible']);
    exit;
}

// Procesar formulario de contacto
if ($_SERVER['REQUEST_METHOD'] == 'POST' && !isset($_POST['ajax_action'])) {
    $nombres_apellidos = trim($_POST['nombres_apellidos']);
    $celular = trim($_POST['celular']);
    $ciudad = trim($_POST['ciudad']);
    $fecha_registro = $_POST['fecha_registro'];
    $estado = $has_estado ? ($_POST['estado'] ?? 'Frio') : null;
    $observaciones = $has_observaciones ? ($_POST['observaciones'] ?? '') : null;
    
    if (empty($nombres_apellidos) || empty($celular) || empty($ciudad) || empty($fecha_registro)) {
        $message = 'Todos los campos obligatorios deben ser completados.';
        $message_type = 'error';
    } else {
        try {
            if (isset($_POST['contacto_id']) && !empty($_POST['contacto_id'])) {
                // Actualizar contacto existente
                $contacto_id = (int)$_POST['contacto_id'];
                
                $campos = "nombres_apellidos = ?, celular = ?, ciudad = ?, fecha_registro = ?";
                $valores = [$nombres_apellidos, $celular, $ciudad, $fecha_registro];
                
                if ($has_estado) {
                    $campos .= ", estado = ?";
                    $valores[] = $estado;
                }
                
                if ($has_observaciones) {
                    $campos .= ", observaciones = ?";
                    $valores[] = $observaciones;
                }
                
                $valores[] = $contacto_id;
                
                $query = "UPDATE contactos SET $campos WHERE id = ?";
                $stmt = $db->prepare($query);
                
                if ($stmt->execute($valores)) {
                    $message = 'Contacto actualizado exitosamente.';
                    $message_type = 'success';
                } else {
                    $message = 'Error al actualizar el contacto.';
                    $message_type = 'error';
                }
            } else {
                // Crear nuevo contacto
                $campos = "usuario_id, nombres_apellidos, celular, ciudad, fecha_registro";
                $placeholders = "?, ?, ?, ?, ?";
                $valores = [$_SESSION['user_id'], $nombres_apellidos, $celular, $ciudad, $fecha_registro];
                
                if ($has_estado) {
                    $campos .= ", estado";
                    $placeholders .= ", ?";
                    $valores[] = $estado;
                }
                
                if ($has_observaciones) {
                    $campos .= ", observaciones";
                    $placeholders .= ", ?";
                    $valores[] = $observaciones;
                }
                
                $query = "INSERT INTO contactos ($campos) VALUES ($placeholders)";
                $stmt = $db->prepare($query);
                
                if ($stmt->execute($valores)) {
                    $message = 'Contacto creado exitosamente.';
                    $message_type = 'success';
                } else {
                    $message = 'Error al crear el contacto.';
                    $message_type = 'error';
                }
            }
        } catch (Exception $e) {
            $message = 'Error: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
}

// Eliminar contacto
if (isset($_GET['delete'])) {
    try {
        $contacto_id = (int)$_GET['delete'];
        $query = "DELETE FROM contactos WHERE id = ? AND usuario_id = ?";
        $stmt = $db->prepare($query);
        
        if ($stmt->execute([$contacto_id, $_SESSION['user_id']])) {
            $message = 'Contacto eliminado exitosamente.';
            $message_type = 'success';
        } else {
            $message = 'Error al eliminar el contacto.';
            $message_type = 'error';
        }
    } catch (Exception $e) {
        $message = 'Error: ' . $e->getMessage();
        $message_type = 'error';
    }
}

// Obtener contactos con búsqueda y paginación
$search = isset($_GET['search']) ? trim($_GET['search']) : '';
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$per_page = 10;
$offset = ($page - 1) * $per_page;

$where_conditions = ["usuario_id = ?"];
$params = [$_SESSION['user_id']];

if (!empty($search)) {
    $where_conditions[] = "(nombres_apellidos LIKE ? OR celular LIKE ? OR ciudad LIKE ?)";
    $search_param = "%$search%";
    $params[] = $search_param;
    $params[] = $search_param;
    $params[] = $search_param;
}

$where_clause = implode(' AND ', $where_conditions);

// Contar total de contactos
$count_query = "SELECT COUNT(*) FROM contactos WHERE $where_clause";
$count_stmt = $db->prepare($count_query);
$count_stmt->execute($params);
$total_contactos = $count_stmt->fetchColumn();
$total_pages = ceil($total_contactos / $per_page);

// Obtener contactos
$campos_select = "id, nombres_apellidos, celular, ciudad, fecha_registro";
if ($has_estado) $campos_select .= ", estado";
if ($has_observaciones) $campos_select .= ", observaciones";

$query = "SELECT $campos_select FROM contactos WHERE $where_clause ORDER BY fecha_registro DESC LIMIT $per_page OFFSET $offset";
$stmt = $db->prepare($query);
$stmt->execute($params);
$contactos = $stmt->fetchAll();

// Obtener etiquetas disponibles (solo si la tabla existe)
$etiquetas = [];
if ($has_etiquetas) {
    try {
        $query = "SELECT * FROM etiquetas ORDER BY nombre";
        $stmt = $db->prepare($query);
        $stmt->execute();
        $etiquetas = $stmt->fetchAll();
    } catch (Exception $e) {
        // Ignorar errores
    }
}
?>

<?php
require_once 'settings.php';
$systemSettings = getSystemSettings();
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestión de Contactos - CRM</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link href="dynamic_styles.php" rel="stylesheet" type="text/css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: #f8f9fa;
            line-height: 1.6;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }
        
        .header {
            background: linear-gradient(135deg, #6366f1 0%, #8b5cf6 100%);
            color: white;
            padding: 2rem;
            border-radius: 10px;
            margin-bottom: 2rem;
            box-shadow: 0 4px 20px rgba(99, 102, 241, 0.15);
        }
        
        .alert {
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 5px;
            font-weight: 500;
        }
        
        .alert.success {
            background-color: #d4edda;
            color: #155724;
            border: 1px solid #c3e6cb;
        }
        
        .alert.error {
            background-color: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
        
        .migration-notice {
            background: linear-gradient(135deg, #ffeaa7 0%, #fab1a0 100%);
            color: #2d3436;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
            text-align: center;
        }
        
        .btn {
            background-color: #007bff;
            color: white;
            padding: 10px 20px;
            border: none;
            border-radius: 5px;
            cursor: pointer;
            text-decoration: none;
            display: inline-block;
            margin: 5px;
            font-size: 14px;
        }
        
        .btn:hover {
            background-color: #0056b3;
        }
        
        .btn-success { background-color: #28a745; }
        .btn-success:hover { background-color: #218838; }
        
        .btn-danger { background-color: #dc3545; }
        .btn-danger:hover { background-color: #c82333; }
        
        .form-group {
            margin-bottom: 15px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 5px;
            font-weight: 500;
        }
        
        .form-control {
            width: 100%;
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 5px;
            font-size: 14px;
        }
        
        .table {
            width: 100%;
            border-collapse: collapse;
            background: white;
            border-radius: 10px;
            overflow: hidden;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .table th,
        .table td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #ddd;
        }
        
        .table th {
            background-color: #f8f9fa;
            font-weight: 600;
        }
        
        .pagination {
            display: flex;
            justify-content: center;
            margin-top: 20px;
        }
        
        .pagination a {
            padding: 8px 12px;
            margin: 0 2px;
            text-decoration: none;
            border: 1px solid #ddd;
            color: #007bff;
            border-radius: 3px;
        }
        
        .pagination a:hover,
        .pagination a.active {
            background-color: #007bff;
            color: white;
        }
        
        .search-form {
            background: white;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        
        .form-row {
            display: flex;
            gap: 15px;
            align-items: end;
        }
        
        .form-row .form-group {
            flex: 1;
        }
        
        .status-badge {
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 500;
        }
        
        .status-frio { background-color: #e3f2fd; color: #1976d2; }
        .status-tibio { background-color: #fff3e0; color: #f57c00; }
        .status-caliente { background-color: #ffebee; color: #d32f2f; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <div style="display: flex; justify-content: space-between; align-items: center;">
                <div>
                    <h1>📞 Gestión de Contactos</h1>
                    <p><?php echo htmlspecialchars($systemSettings['company_name']); ?> - Versión Segura</p>
                </div>
                <div style="display: flex; gap: 10px;">
                    <a href="dashboard.php" class="btn" style="background: rgba(255,255,255,0.2); color: white;">🏠 Dashboard</a>
                    <a href="admin.php" class="btn" style="background: rgba(255,255,255,0.2); color: white;">⚙️ Admin</a>
                    <a href="logout.php" class="btn" style="background: rgba(255,255,255,0.2); color: white;">🚪 Salir</a>
                </div>
            </div>
        </div>
        
        <?php if (!$has_estado || !$has_observaciones || !$has_etiquetas || !$has_historial): ?>
        <div class="migration-notice">
            <h3>⚠️ Funcionalidades Limitadas</h3>
            <p>Para acceder a todas las funcionalidades (estados, etiquetas, historial), necesitas ejecutar la migración de base de datos.</p>
            <a href="run_migration.html" class="btn btn-success">🔄 Ejecutar Migración</a>
            <a href="migrate_database.php" class="btn" target="_blank">📋 Ver Migración</a>
        </div>
        <?php endif; ?>
        
        <?php if ($message): ?>
            <div class="alert <?php echo $message_type; ?>">
                <?php echo htmlspecialchars($message); ?>
            </div>
        <?php endif; ?>
        
        <!-- Formulario de búsqueda -->
        <div class="search-form">
            <form method="GET" class="form-row">
                <div class="form-group">
                    <label for="search">Buscar contactos:</label>
                    <input type="text" id="search" name="search" class="form-control" 
                           placeholder="Nombre, teléfono o ciudad..." 
                           value="<?php echo htmlspecialchars($search); ?>">
                </div>
                <div class="form-group">
                    <button type="submit" class="btn">🔍 Buscar</button>
                    <a href="contactos_safe.php" class="btn" style="background-color: #6c757d;">🔄 Limpiar</a>
                </div>
            </form>
        </div>
        
        <!-- Formulario de contacto -->
        <div class="search-form">
            <h3>➕ Agregar/Editar Contacto</h3>
            <form method="POST">
                <input type="hidden" id="contacto_id" name="contacto_id">
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="nombres_apellidos">Nombres y Apellidos *</label>
                        <input type="text" id="nombres_apellidos" name="nombres_apellidos" class="form-control" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="celular">Celular *</label>
                        <input type="text" id="celular" name="celular" class="form-control" required>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="ciudad">Ciudad *</label>
                        <input type="text" id="ciudad" name="ciudad" class="form-control" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="fecha_registro">Fecha de Registro *</label>
                        <input type="date" id="fecha_registro" name="fecha_registro" class="form-control" required>
                    </div>
                </div>
                
                <?php if ($has_estado): ?>
                <div class="form-row">
                    <div class="form-group">
                        <label for="estado">Estado</label>
                        <select id="estado" name="estado" class="form-control">
                            <option value="Frio">❄️ Frío</option>
                            <option value="Tibio">🌡️ Tibio</option>
                            <option value="Caliente">🔥 Caliente</option>
                        </select>
                    </div>
                </div>
                <?php endif; ?>
                
                <?php if ($has_observaciones): ?>
                <div class="form-group">
                    <label for="observaciones">Observaciones</label>
                    <textarea id="observaciones" name="observaciones" class="form-control" rows="3" 
                              placeholder="Notas adicionales sobre el contacto..."></textarea>
                </div>
                <?php endif; ?>
                
                <div class="form-group">
                    <button type="submit" class="btn btn-success">💾 Guardar Contacto</button>
                    <button type="button" class="btn" onclick="limpiarFormulario()">🔄 Limpiar</button>
                </div>
            </form>
        </div>
        
        <!-- Lista de contactos -->
        <div class="search-form">
            <h3>📋 Lista de Contactos (<?php echo $total_contactos; ?> total)</h3>
            
            <?php if (empty($contactos)): ?>
                <p style="text-align: center; color: #666; padding: 40px;">
                    <?php echo empty($search) ? '📭 No hay contactos registrados.' : '🔍 No se encontraron contactos con esa búsqueda.'; ?>
                </p>
            <?php else: ?>
                <div style="overflow-x: auto;">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Nombre</th>
                                <th>Celular</th>
                                <th>Ciudad</th>
                                <th>Fecha</th>
                                <?php if ($has_estado): ?><th>Estado</th><?php endif; ?>
                                <?php if ($has_observaciones): ?><th>Observaciones</th><?php endif; ?>
                                <th>Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($contactos as $contacto): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($contacto['nombres_apellidos']); ?></td>
                                    <td><?php echo htmlspecialchars($contacto['celular']); ?></td>
                                    <td><?php echo htmlspecialchars($contacto['ciudad']); ?></td>
                                    <td><?php echo date('d/m/Y', strtotime($contacto['fecha_registro'])); ?></td>
                                    
                                    <?php if ($has_estado): ?>
                                    <td>
                                        <span class="status-badge status-<?php echo strtolower($contacto['estado']); ?>">
                                            <?php 
                                            $iconos = ['Frio' => '❄️', 'Tibio' => '🌡️', 'Caliente' => '🔥'];
                                            echo $iconos[$contacto['estado']] . ' ' . $contacto['estado'];
                                            ?>
                                        </span>
                                    </td>
                                    <?php endif; ?>
                                    
                                    <?php if ($has_observaciones): ?>
                                    <td>
                                        <?php 
                                        $obs = $contacto['observaciones'];
                                        echo $obs ? (strlen($obs) > 50 ? substr(htmlspecialchars($obs), 0, 50) . '...' : htmlspecialchars($obs)) : '-';
                                        ?>
                                    </td>
                                    <?php endif; ?>
                                    
                                    <td>
                                        <button class="btn" style="background-color: #28a745; font-size: 12px;" 
                                                onclick="editarContacto(<?php echo htmlspecialchars(json_encode($contacto)); ?>)">
                                            ✏️ Editar
                                        </button>
                                        <a href="?delete=<?php echo $contacto['id']; ?>" 
                                           class="btn btn-danger" style="font-size: 12px;"
                                           onclick="return confirm('¿Estás seguro de eliminar este contacto?')">
                                            🗑️ Eliminar
                                        </a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                
                <!-- Paginación -->
                <?php if ($total_pages > 1): ?>
                    <div class="pagination">
                        <?php for ($i = 1; $i <= $total_pages; $i++): ?>
                            <a href="?page=<?php echo $i; ?><?php echo $search ? '&search=' . urlencode($search) : ''; ?>" 
                               class="<?php echo $i == $page ? 'active' : ''; ?>">
                                <?php echo $i; ?>
                            </a>
                        <?php endfor; ?>
                    </div>
                <?php endif; ?>
            <?php endif; ?>
        </div>
        
        <div style="text-align: center; margin-top: 30px; color: #666;">
            <p>🔗 <a href="dashboard.php">Ir al Dashboard</a> | <a href="admin.php">Panel Admin</a> | <a href="logout.php">Cerrar Sesión</a></p>
        </div>
    </div>
    
    <script>
        function editarContacto(contacto) {
            document.getElementById('contacto_id').value = contacto.id;
            document.getElementById('nombres_apellidos').value = contacto.nombres_apellidos;
            document.getElementById('celular').value = contacto.celular;
            document.getElementById('ciudad').value = contacto.ciudad;
            document.getElementById('fecha_registro').value = contacto.fecha_registro;
            
            <?php if ($has_estado): ?>
            if (contacto.estado) {
                document.getElementById('estado').value = contacto.estado;
            }
            <?php endif; ?>
            
            <?php if ($has_observaciones): ?>
            if (contacto.observaciones) {
                document.getElementById('observaciones').value = contacto.observaciones;
            }
            <?php endif; ?>
            
            // Scroll al formulario
            document.querySelector('.search-form h3').scrollIntoView({ behavior: 'smooth' });
        }
        
        function limpiarFormulario() {
            document.getElementById('contacto_id').value = '';
            document.getElementById('nombres_apellidos').value = '';
            document.getElementById('celular').value = '';
            document.getElementById('ciudad').value = '';
            document.getElementById('fecha_registro').value = new Date().toISOString().split('T')[0];
            
            <?php if ($has_estado): ?>
            document.getElementById('estado').value = 'Frio';
            <?php endif; ?>
            
            <?php if ($has_observaciones): ?>
            document.getElementById('observaciones').value = '';
            <?php endif; ?>
        }
        
        // Establecer fecha actual por defecto
        document.getElementById('fecha_registro').value = new Date().toISOString().split('T')[0];
    </script>
</body>
</html>